
#### PLOTS AND PLOT DATA FOR GARVIN, KEARNEY, ROSE (2021)

library(dplyr)
library(tidyr)
library(data.table)
library(ggplot2)
library(gridExtra)
library(magrittr)

rm(list=ls())
options(error=recover)
options(error=NULL)


data <- read.csv('data_for_macropru_analysis.csv')

# Add in variables for total IO and PI approvals
data <- cbind(data, Appr_IO=NA, Appr_PI=NA)
data$Appr_IO <- ifelse(is.na(data$Appr_Inv_IO) & is.na(data$Appr_Occ_IO), NA, 
                       rowSums(data[, c('Appr_Inv_IO', 'Appr_Occ_IO')], na.rm=TRUE))
data$Appr_PI <- data$Appr_Tot_Tot - ifelse(is.na(data$Appr_IO), 0, data$Appr_IO)
  



widen_F <- function(data, variable, grth_interval, years, drop_if_period_NA=NULL) {
  y <- data[, c('period', 'bank', variable)] %>%
    pivot_wider(id_cols=period, names_from=bank, values_from=all_of(variable)) %>%
    data.frame
  # Convert to growth rates over 'grth_interval' quarters
  if(grth_interval > 0) {
    y <- cbind(period=y$period[-(1:grth_interval)], 
               tail(y[, -1], -grth_interval)/head(y[, -1], -grth_interval))
  }
  # Drop banks with NA/Inf in the period specified
  if(length(drop_if_period_NA) > 0) {
    keep <- c(TRUE, !is.na(y[y$period == drop_if_period_NA, -1]) & 
                abs(y[y$period == drop_if_period_NA, -1]) < Inf)
    y <- y[, keep]
  }
  y <- y[substr(y$period, 1, 4) %in% years, ]
  y
}

scatter_2p_F <- function(data_x, dates_x, data_y, dates_y, x_label=NULL, y_label=NULL,
                         hline=NULL, vline=NULL, same_ax_lim=FALSE, line_excl_maj=FALSE,
                         nofitline=FALSE) {
  # Other colours
  chart_F <- function() {
    par(xpd=FALSE)

    banks <- intersect(colnames(data_y), colnames(data_x))
    data_x <- data_x[, banks]
    data_y <- data_y[, banks]

    majcol <- 2
    noncol <- 4
    majors <- ifelse(colnames(data_x)[-1] %in% c('ANZ', 'CBA', 'NAB', 'WBC'),
                     majcol, noncol)

    scatdata <- vector('list', 2)
    for(i in 1:2) scatdata[[i]] <-
      cbind(x=as.numeric(data_x[data_x$period == dates_x[i], -1]),
            y=as.numeric(data_y[data_y$period == dates_y[i], -1]))
    # Remove any NA or Inf from all plots
    rmv_NA <- unique(unlist(sapply(scatdata,
                                   function(x) which(is.na(rowSums(x)) |
                                                     abs(rowSums(x)) > 100))))
    if(length(rmv_NA) > 0) scatdata <- lapply(scatdata, function(x) x[-rmv_NA, ])
    
    par(mfrow=c(1, 2))
    xlim=range(unlist(lapply(scatdata, function(xx) xx[, 1])), na.rm=TRUE)
    ylim=range(unlist(lapply(scatdata, function(xx) xx[, 2])), na.rm=TRUE)
    if(same_ax_lim) {
      xlim <- range(c(xlim, ylim))
      ylim <- range(c(xlim, ylim))
    }
    plot(scatdata[[1]][, 1], scatdata[[1]][, 2], ylim=ylim, xlim=xlim, pch=4,
         col=majors, xlab=x_label, ylab=y_label, main=dates_y[1])
    if(line_excl_maj) d1 <- scatdata[[1]][majors == noncol, ] else d1 <- scatdata[[1]]
    if(!nofitline) abline(lm(d1[, 2] ~ d1[, 1]), col='gray')
    abline(v=vline, h=hline)

    plot(scatdata[[2]][, 1], scatdata[[2]][, 2], ylim=ylim, xlim=xlim, pch=4,
         col=majors, xlab=x_label, ylab=y_label, main=dates_y[2])
    if(line_excl_maj) d2 <- scatdata[[2]][majors == noncol, ] else d2 <- scatdata[[2]]
    if(!nofitline) abline(lm(d2[, 2] ~ d2[, 1]), col='gray')
    abline(v=vline, h=hline)
    
    # browser()
    # outdata <- c()
    # outdata$d1 <- cbind(x=scatdata[[1]][, 1], y=scatdata[[1]][, 2], maj=majors)
    # outdata$ols1 <- summary(lm(d1[, 2] ~ d1[, 1]))
    # outdata$d2 <- cbind(x=scatdata[[2]][, 1], y=scatdata[[2]][, 2], maj=majors)
    # outdata$ols2 <- summary(lm(d2[, 2] ~ d2[, 1]))
    
  }
  return(chart_F())
}


##########################################################################################
### Producing output used in the paper
##########################################################################################
### GRAPHS

# Summary stats charts
# Investor and IO shares of total approvals
summary_charts_F <- function(data) {
  agg_appr <- data %>% data.table %>% 
    .[, .(inv=sum(Appr_Inv_Tot, na.rm=TRUE), occ=sum(Appr_Occ_Tot, na.rm=TRUE), 
          invio=sum(Appr_Inv_IO, na.rm=TRUE), occio=sum(Appr_Occ_IO, na.rm=TRUE)), 
      by=period] %>%
    .[, total := inv + occ] %>%
    .[, c('invpi_shr', 'invio_shr', 'occpi_shr', 'occio_shr') := 
        list((inv - invio)/total, invio/total, (occ - occio)/total, occio/total)]
  
  appr_chart <- data %>% data.table %>% 
    .[, .(inv=sum(Appr_Inv_Tot, na.rm=TRUE), occ=sum(Appr_Occ_Tot, na.rm=TRUE), 
          invio=sum(Appr_Inv_IO, na.rm=TRUE), occio=sum(Appr_Occ_IO, na.rm=TRUE)), 
      by=period] %>%
    .[, .(period, occpi=(occ - occio)/1e9, occio=occ/1e9, 
          invpi=(occ + inv - invio)/1e9, invio=(inv + occ)/1e9)]
  int_chart <- data %>% data.table %>%
    .[, .(occpi=mean(OOPI_rate, na.rm=TRUE), 
          occio=mean(OOIO_rate_with_replacement, na.rm=TRUE),
          invpi=mean(INVPI_rate, na.rm=TRUE),
          invio=mean(INVIO_rate_with_replacement, na.rm=TRUE)),
      by=period]
  
  macro_chart <- data %>% data.table %>%
    .[, .(gdp_gr=mean(GDP_qpd), hp_gr=mean(housepr_qpd), cr=mean(CR_qe)), 
      by=period]
    
  return(list(appr_chart=appr_chart, int_chart=int_chart, macro_chart=macro_chart))
}

summary_charts_F(data)


#------------------------------------------##--------------------------------------------#
### Investor policy, approvals
inv_pol_appr_chart_F <- function(data) {
  # Aggregate mortgage approvals growth
  agg_appr_F <- function(data) {
    tot_F <- function(x) data.table(period=x$period, appr=rowSums(x[, -1], na.rm=TRUE))
    own <- widen_F(data, 'Appr_Occ_Tot', 0, 2008:2019) %>% tot_F
    inv <- widen_F(data, 'Appr_Inv_Tot', 0, 2008:2019) %>% tot_F
    ownio <- widen_F(data, 'Appr_Occ_IO', 0, 2008:2019) %>% tot_F
    invio <- widen_F(data, 'Appr_Inv_IO', 0, 2008:2019) %>% tot_F
    y <- data.table(date=own$period, own=own$appr, 
                    inv=inv$appr[match(own$period, inv$period)],
                    ownio=ownio$appr[match(own$period, inv$period)],
                    invio=invio$appr[match(own$period, invio$period)])
    y <- y[, `:=`(ownpi=own - ownio, 
                  invpi=inv - invio, 
                  pi=own - ownio + inv - invio, 
                  io=ownio + invio)]
    # Convert to annual growth
    y <- data.table(date=tail(y$date, -4), 100*(tail(y[, -1], -4)/head(y[, -1], -4) - 1))
    y
  }
  appr <- agg_appr_F(data)
  # Number of banks above benchmark
  bal_ann <- widen_F(data, 'Outs_Inv_ClosBal', 4, 2010:2019, '2014-12-31')
  # No observations in q4 2019
  bal_ann <- bal_ann[bal_ann$period != '2019-12-31', ]
  bal_ann[is.na(bal_ann)] <- 1

  date <- bal_ann$period
  num_abv <- apply(bal_ann[, -1], 1, function(x) sum(x > 1.1 & x < Inf))
  graph_data <- data.frame(date, 
                           oo_appr_gr=appr$own[appr$date %in% date],
                           in_appr_gr=appr$inv[appr$date %in% date],
                           num_abv=num_abv[bal_ann$period %in% date])
  return(graph_data)
}
inv_pol_appr_chart_F(data)
#------------------------------------------##--------------------------------------------#


#------------------------------------------##--------------------------------------------#
### Investor policy counterfactuals
#------------------------------------------##--------------------------------------------#
counterfactuals_F <- function(data, levels=FALSE) {
  years <- 2014:2015
  majors <- c('ANZ', 'CBA', 'NAB', 'WBC')
  # Coefficients from regression output
  coefs_inv	<-	c()
  coefs_inv$m$a1	<-	-0.025
  coefs_inv$m$a2	<-	-0.059
  coefs_inv$m$a3	<-	-0.236
  coefs_inv$m$a4	<-	-0.025
  coefs_inv$m$b1	<-	0.019
  coefs_inv$m$b2	<-	-0.27
  
  coefs_inv$n$a1	<-	-0.07
  coefs_inv$n$a2	<-	-0.031
  coefs_inv$n$a3	<-	-0.241
  coefs_inv$n$a4	<-	-0.423
  coefs_inv$n$b1	<-	0.025
  coefs_inv$n$b2	<-	-0.167
  
  coefs_occ	<-	c()
  coefs_occ$m$a1	<-	-0.008
  coefs_occ$m$a2	<-	0.02
  coefs_occ$m$a3	<-	0.144
  coefs_occ$m$a4	<-	0.096
  coefs_occ$m$b1	<-	-0.023
  coefs_occ$m$b2	<-	-0.27
  
  coefs_occ$n$a1	<-	0.029
  coefs_occ$n$a2	<-	0.002
  coefs_occ$n$a3	<-	-0.005
  coefs_occ$n$a4	<-	0.002
  coefs_occ$n$b1	<-	-0.082
  coefs_occ$n$b2	<-	-0.165
  
  # Function for taking individual approvals and generating counterfactuals
  temp_cfl_F <- function(ind_appr, coefs, for_majors) {
    y <- c()
    y$periods <- ind_appr$period
    # Select either majors or others
    if(for_majors) {
      y$lev <- ind_appr[, majors]
      coefs <- coefs$m
    } else {
      y$lev <- ind_appr[, !colnames(ind_appr) %in% majors][, -1]
      coefs <- coefs$n
    }
    # Convert to growth rates
    y$gth <- rbind(rep(NA, ncol(y$lev)), tail(y$lev, -1)/head(y$lev, -1) - 1)
    y$gth_cfl <- y$gth
    y$gth_cfl[y$periods == '2015-03-31', ] <- y$gth[y$periods == '2015-03-31', ] -
      coefs$a1
    y$gth_cfl[y$periods == '2015-06-30', ] <- y$gth[y$periods == '2015-06-30', ] -
      (coefs$a2 + coefs$b1 * coefs$a1)
    y$gth_cfl[y$periods == '2015-09-30', ] <- y$gth[y$periods == '2015-09-30', ] -
      (coefs$a3 + coefs$b1 * coefs$a2 + (coefs$b1^2 + coefs$b2) * coefs$a3)
    y$gth_cfl[y$periods == '2015-12-31', ] <- y$gth[y$periods == '2015-12-31', ] -
      (coefs$a4 + coefs$b1 * coefs$a3 + (coefs$b1^2 + coefs$b2) * coefs$a2 + 
         (coefs$b1^3 + coefs$b2^2) * coefs$a1)
    y$lev_cfl <- y$lev
    for(i in 2:nrow(y$lev_cfl)) y$lev_cfl[i, ] <- y$lev_cfl[i-1, ] * (1 + y$gth_cfl[i, ])
    y$tot_lev <- rowSums(y$lev)
    y$tot_lev_cfl <- rowSums(y$lev_cfl)
    y$tot_gth <- c(NA, tail(y$tot_lev, -1)/head(y$tot_lev, -1) - 1)
    y$tot_gth_cfl <- c(NA, tail(y$tot_lev_cfl, -1)/head(y$tot_lev_cfl, -1) - 1)
    return(y)
  }
  # Run the function on different types of approvals for majors and nonmajors
  inv_app <- widen_F(data, 'Appr_Inv_Tot', grth_interval=0, years=years, 
                     drop_if_period_NA='2013-09-30')
  occ_app <- widen_F(data, 'Appr_Occ_Tot', grth_interval=0, years=years, 
                     drop_if_period_NA='2013-09-30')
  y <- c()
  y$inv_app_maj <- temp_cfl_F(inv_app, coefs_inv, for_majors=TRUE)
  y$occ_app_maj <- temp_cfl_F(occ_app, coefs_occ, for_majors=TRUE)
  y$inv_app_non <- temp_cfl_F(inv_app, coefs_inv, for_majors=FALSE)
  y$occ_app_non <- temp_cfl_F(occ_app, coefs_occ, for_majors=FALSE)
  
  if(!levels) {
    graph_data <- data.frame(period=y$inv_app_maj$periods,
                             act_maj_inv=y$inv_app_maj$tot_gth * 100,
                             cfl_maj_inv=y$inv_app_maj$tot_gth_cfl * 100,
                             act_non_inv=y$inv_app_non$tot_gth * 100,
                             cfl_non_inv=y$inv_app_non$tot_gth_cfl * 100,
                             act_maj_occ=y$occ_app_maj$tot_gth * 100,
                             cfl_maj_occ=y$occ_app_maj$tot_gth_cfl * 100,
                             act_non_occ=y$occ_app_non$tot_gth * 100,
                             cfl_non_occ=y$occ_app_non$tot_gth_cfl * 100)
  } else {
    graph_data <- data.frame(period=y$inv_app_maj$periods,
                             act_maj_inv=y$inv_app_maj$tot_lev / 1e9,
                             cfl_maj_inv=y$inv_app_maj$tot_lev_cfl / 1e9,
                             act_non_inv=y$inv_app_non$tot_lev / 1e9,
                             cfl_non_inv=y$inv_app_non$tot_lev_cfl / 1e9,
                             act_maj_occ=y$occ_app_maj$tot_lev / 1e9,
                             cfl_maj_occ=y$occ_app_maj$tot_lev_cfl / 1e9,
                             act_non_occ=y$occ_app_non$tot_lev / 1e9,
                             cfl_non_occ=y$occ_app_non$tot_lev_cfl / 1e9)
  }
  
  return(graph_data)
}
x <- counterfactuals_F(data, levels=TRUE)
x
#------------------------------------------##--------------------------------------------#


#------------------------------------------##--------------------------------------------#
### Investor policy interest rates v approvals
#------------------------------------------##--------------------------------------------#
approvals_v_rates_scatter_F <- function(data) {
  majors <- c('ANZ', 'CBA', 'NAB', 'WBC')
  invint <- widen_F(data, 'INVPI_rate', 0, 2015:2016, '2013-12-31')
  invintd <- data.frame(period=invint$period[-1], 
                        tail(invint[, -1], -1) - head(invint[, -1], -1))
  invapp <- widen_F(data, 'Appr_Inv_Tot', 0, 2015:2016, '2013-12-31')
  invappgr <- widen_F(data, 'Appr_Inv_Tot', 1, 2015:2016, '2013-12-31')
  
  banks <- intersect(colnames(invint)[-1], colnames(invappgr)[-1])
  
  # caluculate the averages
  av_F <- function(x, dates) x[x$period %in% dates, banks] %>% colMeans %>% unlist
  intavq34 <- av_F(invint, c('2015-09-30', '2015-12-31'))
  intavq3412 <- av_F(invint, c('2015-09-30', '2015-12-31', '2016-03-31', '2016-06-30'))
  appavq34 <- av_F(invapp, c('2015-09-30', '2015-12-31'))
  appavq3412 <- av_F(invapp, c('2015-09-30', '2015-12-31', '2016-03-31', '2016-06-30'))
  
  # Put together the graph data
  scat <- 
    data.frame(banks,
               intdq3=unlist(100*invintd[invintd$period == '2015-09-30', banks]),
               apgrq3=unlist(100*(invappgr[invappgr$period == '2015-09-30', banks] - 1)),
               intdq4=unlist(100*invintd[invintd$period == '2015-12-31', banks]),
               apgrq4=unlist(100*(invappgr[invappgr$period == '2015-12-31', banks] - 1)),
               intdavq4=100*(intavq34 - 
                               unlist(invint[invint$period == '2015-06-30', banks])),
               apgravq4=100*(appavq34 / 
                               unlist(invapp[invapp$period == '2015-06-30', banks]) - 1),
               intdavq2=100*(intavq3412 - 
                               unlist(invint[invint$period == '2015-06-30', banks])),
               apgravq2=100*(appavq3412 / 
                               unlist(invapp[invapp$period == '2015-06-30', banks]) - 1),
               majorcol=ifelse(banks %in% majors, 2, 4))
  
  par(mfrow=c(2, 2))
  xlim <- scat[, grepl('^intd', colnames(scat))] %>% unlist %>% range
  ylim <- scat[, grepl('^apgr', colnames(scat))] %>% unlist %>% range
  ylab1 <- 'Investor commitments growth quarterly (%)'
  xlab1 <- 'Investor rate change quarterly (bps)'
  ylab2 <- 'Investor commitments growth, average since Q2 2015 (%)'
  xlab2 <- 'Investor rate change, average since Q2 2015 (bps)'
  
  plot(scat$intdq3, scat$apgrq3, ylim=ylim, xlim=xlim, pch=4, 
       col=scat$majorcol, xlab=xlab1, ylab=ylab1, main='Q3 2015')
  abline(lm(scat$apgrq3 ~ scat$intdq3), col='gray')
  legend(y=60, x=40, legend=c('Large', 'Mid-sized'), 
         col=c('red', 'blue'), pch=c(4, 4), box.lty=1, bg='transparent') 
  plot(scat$intdq4, scat$apgrq4, ylim=ylim, xlim=xlim, pch=4, 
       col=scat$majorcol, xlab=xlab1, ylab=ylab1, main='Q4 2015')
  abline(lm(scat$apgrq4 ~ scat$intdq4), col='gray')
  plot(scat$intdavq4, scat$apgravq4, ylim=ylim, xlim=xlim, pch=4, 
       col=scat$majorcol, xlab=xlab2, ylab=ylab2, 
       main='Q3 and Q4 2015 average')
  abline(lm(scat$apgravq4 ~ scat$intdavq4), col='gray')
  plot(scat$intdavq2, scat$apgravq2, ylim=ylim, xlim=xlim, pch=4, 
       col=scat$majorcol, xlab=xlab2, ylab=ylab2, 
       main='Q3 2015 to Q2 2016 average')
  abline(lm(scat$apgravq2 ~ scat$intdavq2), col='gray')
  
  # Also return investor approvals growth to show some of the 2016 behaviour
  invappgr[, -1] <- 100*(invappgr[, -1] - 1)
  return(list(scatdata=scat, invapprgr=invappgr))
}
approvals_v_rates_scatter_F(data)
approvals_v_rates_scatter_F(data)$scatdata
#------------------------------------------##--------------------------------------------#
 

#------------------------------------------##--------------------------------------------#
# POlicy 1 treatment intensity visualisation
#------------------------------------------##--------------------------------------------#

inv_pol_het_chart_F <- function(data) {
  inv_ann <- widen_F(data, 'Outs_Inv_ClosBal', 4, 2015:2016, '2013-12-31')
  inv_ann[, -1] <- 100*(inv_ann[, -1] - 1)
  app_qrt <- widen_F(data, 'Appr_Inv_Tot', 1, 2015:2016, '2013-12-31')
  #app_qrt <- data.frame(period=app_qrt$period, 
  #                      100*rbind(NA, tail(app_qrt[, -1], -1) - head(app_qrt[, -1], -1)))
  app_qrt[, -1] <- (app_qrt[, -1] - 1)*100
  app_qrt <- app_qrt[, colnames(inv_ann)]
  
  # Starting in Q3 2015
  scatter_2p_F(data_x=inv_ann, dates_x=c('2015-03-31', '2015-06-30'),
               data_y=app_qrt, dates_y=c('2015-09-30', '2015-12-31'),
               x_label='Year-ended investor credit growth (t-2, %)', 
               y_label=expression(paste('Investor commitments growth (%, quarterly)')), 
               vline=10)
  legend(y=25, x=40, legend=c('Large', 'Mid-sized'), 
         col=c('red', 'blue'), pch=c(4, 4), box.lty=1, bg='transparent') 
  
  # Also return investor approvals growth over the next couple of quarters
  return(list(approvals_growth=app_qrt, 
              rate_chng=widen_F(data, 'INVPI_rate', 0, 2014:2015, '2013-12-31')))
}
inv_pol_het_chart_F(data)


#------------------------------------------##--------------------------------------------#
# POlicy 2 aggregates
#------------------------------------------##--------------------------------------------#
io_pol_appr_chart_F <- function(data) {
  # Aggregate mortgage approvals growth
  agg_appr_F <- function(data) {
    tot_F <- function(x) data.table(period=x$period, appr=rowSums(x[, -1], na.rm=TRUE))
    own <- widen_F(data, 'Appr_Occ_Tot', 0, 2008:2019) %>% tot_F
    inv <- widen_F(data, 'Appr_Inv_Tot', 0, 2008:2019) %>% tot_F
    ownio <- widen_F(data, 'Appr_Occ_IO', 0, 2008:2019) %>% tot_F
    invio <- widen_F(data, 'Appr_Inv_IO', 0, 2008:2019) %>% tot_F
    y <- data.table(date=own$period, own=own$appr, 
                    inv=inv$appr[match(own$period, inv$period)],
                    ownio=ownio$appr[match(own$period, inv$period)],
                    invio=invio$appr[match(own$period, invio$period)])
    y <- y[, `:=`(ownpi=own - ownio, 
                  invpi=inv - invio, 
                  pi=own - ownio + inv - invio, 
                  io=ownio + invio)]
    # Convert to annual growth
    y <- data.table(date=tail(y$date, -4), 100*(tail(y[, -1], -4)/head(y[, -1], -4) - 1))
    y
  }
  appr <- agg_appr_F(data)
  # Number of banks above IO benchmark
  invio <- widen_F(data, 'Appr_Inv_IO', 0, 2010:2019, '2016-03-31')
  ownio <- widen_F(data, 'Appr_Occ_IO', 0, 2010:2019, '2016-03-31')
  owntot <- widen_F(data, 'Appr_Occ_Tot', 0, 2010:2019, '2016-03-31')
  invtot <- widen_F(data, 'Appr_Inv_Tot', 0, 2010:2019, '2016-03-31')
  banks <- intersect(colnames(invio), colnames(ownio))
  banks <- banks[banks != 'period']
  sum(!colnames(ownio[, -1]) %in% banks)
  # Calculate IO shares for banks
  y <- data[, c('period', 'bank', 
                'Appr_Occ_Tot', 'Appr_Inv_Tot', 'Appr_Occ_IO', 'Appr_Inv_IO')] %>%
    data.table %>% .[, list(period=period, bank=bank, tot=Appr_Occ_Tot + Appr_Inv_Tot, 
                            io=Appr_Occ_IO + Appr_Inv_IO)] %>%
    .[, ioshare := io/tot] %>% 
    dcast(period ~ bank, value.var='ioshare') %>% data.frame
  y[is.na(y)] <- 0
  num_abv_io <- data.frame(period=y$period, 
                           num_abv=apply(y[, -1], 1, function(x) sum(x > 0.3)))
  date <- appr$date
  graph_data <- data.frame(date, 
                           pi_appr_gr=appr$pi[appr$date %in% date],
                           io_appr_gr=appr$io[appr$date %in% date],
                           num_abv=num_abv_io$num_abv[num_abv_io$period %in% date])
  return(graph_data)
}
io_pol_appr_chart_F(data)
#------------------------------------------##------------------------------------------#


#------------------------------------------##--------------------------------------------#
### IO policy counterfactuals
#------------------------------------------##--------------------------------------------#
counterfactuals_F <- function(data, levels=FALSE) {
  years <- 2016:2018
  majors <- c('ANZ', 'CBA', 'NAB', 'WBC')
  # Coefficients
  coefs_io	<-	c()	
  coefs_io$m$a1	<-	-0.324
  coefs_io$m$a2	<-	-0.476
  coefs_io$m$a3	<-	-0.177
  coefs_io$m$a4	<-	-0.011
  coefs_io$m$b1	<-	-0.201
  coefs_io$m$b2	<-	-0.191
  
  coefs_io$n$a1	<-	-0.27
  coefs_io$n$a2	<-	-0.277
  coefs_io$n$a3	<-	-0.314
  coefs_io$n$a4	<-	0.273
  coefs_io$n$b1	<-	-0.168
  coefs_io$n$b2	<-	-0.073
  
  coefs_pi	<-	c()	
  coefs_pi$m$a1	<-	0.126
  coefs_pi$m$a2	<-	0.172
  coefs_pi$m$a3	<-	0.028
  coefs_pi$m$a4	<-	-0.004
  coefs_pi$m$b1	<-	0.011
  coefs_pi$m$b2	<-	-0.277
  
  coefs_pi$n$a1	<-	-0.149
  coefs_pi$n$a2	<-	0.027
  coefs_pi$n$a3	<-	0.008
  coefs_pi$n$a4	<-	-0.043
  coefs_pi$n$b1	<-	-0.211
  coefs_pi$n$b2	<-	-0.044
  
  # Function for taking individual approvals and generating counterfactuals
  temp_cfl_F <- function(ind_appr, coefs, for_majors) {
    y <- c()
    y$periods <- ind_appr$period
    if(for_majors) {
      y$lev <- ind_appr[, majors]
      coefs <- coefs$m
    } else {
      y$lev <- ind_appr[, !colnames(ind_appr) %in% majors][, -1]
      coefs <- coefs$n
    }
    y$gth <- rbind(rep(NA, ncol(y$lev)), tail(y$lev, -1)/head(y$lev, -1) - 1)
    y$gth_cfl <- y$gth
    y$gth_cfl[y$periods == '2017-06-30', ] <- y$gth[y$periods == '2017-06-30', ] -
      coefs$a1
    y$gth_cfl[y$periods == '2017-09-30', ] <- y$gth[y$periods == '2017-09-30', ] -
      (coefs$a2 + coefs$b1 * coefs$a1)
    y$gth_cfl[y$periods == '2017-12-31', ] <- y$gth[y$periods == '2017-12-31', ] -
      (coefs$a3 + coefs$b1 * coefs$a2 + (coefs$b1^2 + coefs$b2) * coefs$a3)
    y$gth_cfl[y$periods == '2018-03-31', ] <- y$gth[y$periods == '2018-03-31', ] -
      (coefs$a4 + coefs$b1 * coefs$a3 + (coefs$b1^2 + coefs$b2) * coefs$a2 + 
         (coefs$b1^3 + coefs$b2^2) * coefs$a1)
    y$lev_cfl <- y$lev
    for(i in 2:nrow(y$lev_cfl)) y$lev_cfl[i, ] <- y$lev_cfl[i-1, ] * (1 + y$gth_cfl[i, ])
    y$tot_lev <- rowSums(y$lev)
    y$tot_lev_cfl <- rowSums(y$lev_cfl)
    y$tot_gth <- c(NA, tail(y$tot_lev, -1)/head(y$tot_lev, -1) - 1)
    y$tot_gth_cfl <- c(NA, tail(y$tot_lev_cfl, -1)/head(y$tot_lev_cfl, -1) - 1)
    return(y)
  }
  # Run the function on different types of approvals for majors and nonmajors
  io_app <- widen_F(data, 'Appr_IO', grth_interval=0, years=years, 
                    drop_if_period_NA='2016-03-31')
  
  pi_app <- widen_F(data, 'Appr_PI', grth_interval=0, years=years, 
                    drop_if_period_NA='2016-03-31')
  y <- c()
  y$io_app_maj <- temp_cfl_F(io_app, coefs_io, for_majors=TRUE)
  y$pi_app_maj <- temp_cfl_F(pi_app, coefs_pi, for_majors=TRUE)
  y$io_app_non <- temp_cfl_F(io_app, coefs_io, for_majors=FALSE)
  y$pi_app_non <- temp_cfl_F(pi_app, coefs_pi, for_majors=FALSE)
  
  
  if(!levels) {
    graph_data <- data.frame(period=y$io_app_maj$periods,
                             act_maj_io=y$io_app_maj$tot_gth * 100,
                             cfl_maj_io=y$io_app_maj$tot_gth_cfl * 100,
                             act_non_io=y$io_app_non$tot_gth * 100,
                             cfl_non_io=y$io_app_non$tot_gth_cfl * 100,
                             act_maj_pi=y$pi_app_maj$tot_gth * 100,
                             cfl_maj_pi=y$pi_app_maj$tot_gth_cfl * 100,
                             act_non_pi=y$pi_app_non$tot_gth * 100,
                             cfl_non_pi=y$pi_app_non$tot_gth_cfl * 100)
  } else {
    graph_data <- data.frame(period=y$io_app_maj$periods,
                             act_maj_io=y$io_app_maj$tot_lev / 1e9,
                             cfl_maj_io=y$io_app_maj$tot_lev_cfl / 1e9,
                             act_non_io=y$io_app_non$tot_lev / 1e9,
                             cfl_non_io=y$io_app_non$tot_lev_cfl / 1e9,
                             act_maj_pi=y$pi_app_maj$tot_lev / 1e9,
                             cfl_maj_pi=y$pi_app_maj$tot_lev_cfl / 1e9,
                             act_non_pi=y$pi_app_non$tot_lev / 1e9,
                             cfl_non_pi=y$pi_app_non$tot_lev_cfl / 1e9)
  }

  # Manually trim to correct dates
  graph_data <- graph_data[2:9, ]
  
  return(graph_data)
}
x <- counterfactuals_F(data, levels=TRUE)
x
#------------------------------------------##--------------------------------------------#

